<?php

namespace App\Http\Controllers\User;

use App\Models\Account;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class AccountController extends Controller
{
    public function index(Request $request)
    {

        $user = auth()->user();
        if (!$user) {
            abort(403, 'لا يوجد مستخدم.');
        }

        if ($request->ajax()) {

            $draw = $request->get('draw');
            $start = $request->get('start');
            $rowperpage = $request->get('length');
            $searchValue = $request->input('search.value');
            $type = $request->type;
            $from = $request->from;
            $to = $request->to;

            $totalRecords = Account::where('user_id', $user->id)->count();

            $totalRecordswithFilter = Account::where('user_id', $user->id)
                ->when($searchValue, function ($q) use ($searchValue) {
                    $q->where(function ($sub) use ($searchValue) {
                        $sub->where('name', 'like', "%{$searchValue}%")
                            ->orWhere('phone', 'like', "%{$searchValue}%")
                            ->orWhere('bank_account', 'like', "%{$searchValue}%");
                    });
                })->count();

            $accounts = Account::where('user_id', $user->id)
                ->when(!empty($searchValue), function ($q) use ($searchValue) {
                    $q->where(function ($sub) use ($searchValue) {
                        $sub->where('name', 'like', "%{$searchValue}%")
                            ->orWhere('phone', 'like', "%{$searchValue}%")
                            ->orWhere('bank_account', 'like', "%{$searchValue}%");
                    });
                })
                ->when(!empty($type), function ($q) use ($type) {
                    $q->where('type', $type);
                })
                ->when(!empty($from) && !empty($to), function ($q) use ($from, $to) {
                    $q->whereBetween('created_at', [$from . ' 00:00:00', $to . ' 23:59:59']);
                })
                ->when(!empty($from) && empty($to), function ($q) use ($from) {
                    $q->whereDate('created_at', '>=', $from);
                })
                ->when(!empty($to) && empty($from), function ($q) use ($to) {
                    $q->whereDate('created_at', '<=', $to);
                })
                ->skip($start)
                ->take($rowperpage)
                ->orderByDesc('id')
                ->get();


            $records = [];
            foreach ($accounts as $account) {

                $account_type = match ($account->type) {
                        'customer'      => ' <span class="badge bg-success">زبون</span> ',
                        'supplier'      => ' <span class="badge bg-info">مورد</span> ',
                        default     => ' <span class="badge bg-secondary">-</span> ',
                    };

                    if($account->balance > 0){
                        $balance = 'له :<span class="text-success">'. number_format($account->balance, 2) .' </span>';

                    } elseif($account->balance < 0){
                        $balance = 'عليه : <span class="text-warning">'. number_format(abs($account->balance), 2) .'  </span>';

                    } else {
                        $balance = '<span class="text-secondary">'. number_format($account->balance, 2) .'</span>';
                    }

                $records[] = [
                    'name' => $account->name ? $account->name . $account_type:'-',
                    'phone' => $account->phone ? '<a href="tel:' . $account->phone . '">'. $account->phone .'</a>': '-',
                    'balance' => $balance ?? '-',
                    'created_at' => $account->created_at->format('Y-m-d H:i'),
                    'actions' => '<td>
                                            <div class="d-flex gap-1">
                                                <a href="javascript:void(0);" id="account-edit" data-id="' . $account->id . '"
                                                class="btn btn-sm btn-icon btn-outline-primary" title="تعديل">
                                                    تعديل
                                                </a>
                                                <a href="javascript:void(0);" onclick="confirmDeleteAccount(' . $account->id . ')"
                                                class="btn btn-sm btn-icon btn-outline-danger" title="حذف">
                                                    حذف
                                                </a>
                                            </div>
                                        </td>',
                ];
            }

            return response()->json([
                'draw' => intval($draw),
                'iTotalRecords' => $totalRecords,
                'iTotalDisplayRecords' => $totalRecordswithFilter,
                'aaData' => $records,
            ]);
        }

        // ✅ الطلبات العامة لعرض الصفحة (بدون AJAX)
        $accounts = Account::where('user_id', $user->id)->latest()->get();
        $total = $accounts->count();
        return view('user.account.accounts', compact('accounts', 'total'));
    }

    public function create()
    {
        $modalContent = view('user.account.create_account')->render();

        return response()->json(['modalContent' => $modalContent]);
    }


    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'type' => 'required|in:customer,supplier',
            'phone' => 'nullable|string|max:20',
            'bank_account' => 'nullable|string|max:50',
            'phone_account' => 'nullable|string|max:20',
            'name_account' => 'nullable|string|max:100',
            'notes' => 'nullable|string|max:1000',
        ]);

        $validated['user_id'] = auth()->user()->id ?? null;
        Account::create($validated);
        $notification = trans('dash.Created Successfully');
        return response()->json(
            [
                'redirect_url' => route('user.account.index'),
                'notification' => $notification
            ]
        );
    }

    public function edit($id)
    {
        $account = Account::find($id);

        $modalContent = view('user.account.edit_account', compact('account'))->render();
        return response()->json(['modalContent' => $modalContent]);

    }


    public function update(Request $request, $id)
    {
        $account = Account::find($id);
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'type' => 'required|in:customer,supplier',
            'phone' => 'nullable|string|max:20',
            'bank_account' => 'nullable|string|max:50',
            'phone_account' => 'nullable|string|max:20',
            'name_account' => 'nullable|string|max:100',
            'notes' => 'nullable|string|max:1000',
        ]);

        $account->update($validated);
        $notification = trans('dash.Updated Successfully');
        return response()->json(
            [
                'redirect_url' => route('user.account.index'),
                'notification' => $notification
            ]
        );
    }

    public function destroy($id)
    {
        $account = Account::find($id);
        $account->delete();
        $notification = trans('dash.Delete Successfully');
        $notification = array('messege' => $notification, 'alert-type' => 'success');
        return redirect()->route('user.account.index')->with($notification);
    }
    public function getAccount($id)
    {
        $account = \App\Models\Account::find($id);

        if (!$account) {
            return response()->json(['error' => 'الحساب غير موجود'], 404);
        }
        return response()->json([
            'name'           => $account->name,
            'phone'          => $account->phone,
            'bank_account'   => $account->bank_account,
            'phone_account'  => $account->phone_account,
            'name_account'   => $account->name_account,
            'balance'   => $account->balance,
        ]);
    }
}
